document.addEventListener('DOMContentLoaded', () => {
  // Elementos da interface
  const recordBtn = document.getElementById('record-btn');
  const stopBtn = document.getElementById('stop-btn');
  const statusElement = document.getElementById('status');
  const volumeBar = document.getElementById('volume-bar');
  const structuredText = document.getElementById('structured-text');
  const rawText = document.getElementById('raw-text');
  const contextText = document.getElementById('context-text');
  const copyBtn = document.getElementById('copy-btn');
  const clearBtn = document.getElementById('clear-btn');
  const tabButtons = document.querySelectorAll('.tab-btn');
  const tabPanes = document.querySelectorAll('.tab-pane');
  
  // Variáveis para gravação e WebSocket
  let mediaRecorder;
  let audioChunks = [];
  let socket;
  let sessionId;
  let recordingInterval;
  let analyserNode;
  let dataArray;
  let isRecording = false;
  
  // Conectar ao servidor WebSocket
  function connectSocket() {
    socket = io('http://localhost:3000');
    
    socket.on('connect', () => {
      console.log('Conectado ao servidor WebSocket');
      statusElement.textContent = 'Conectado ao servidor';
    });
    
    socket.on('session-started', (data) => {
      sessionId = data.sessionId;
      console.log('Sessão iniciada:', sessionId);
    });
    
    socket.on('transcription-update', (data) => {
      rawText.value = data.transcription;
      if (data.organizedText) {
        structuredText.value = data.organizedText;
      }
      
      // Auto-scroll para a parte inferior
      rawText.scrollTop = rawText.scrollHeight;
      structuredText.scrollTop = structuredText.scrollHeight;
    });
    
    socket.on('error', (error) => {
      console.error('Erro WebSocket:', error);
      statusElement.textContent = 'Erro: ' + error.message;
    });
    
    socket.on('disconnect', () => {
      console.log('Desconectado do servidor WebSocket');
      statusElement.textContent = 'Desconectado do servidor';
    });
  }
  
  // Iniciar gravação
  async function startRecording() {
    try {
      // Solicitar permissão para acessar o microfone
      const stream = await navigator.mediaDevices.getUserMedia({ audio: true });
      
      // Configurar o analisador de áudio para o medidor de volume
      const audioContext = new AudioContext();
      const source = audioContext.createMediaStreamSource(stream);
      analyserNode = audioContext.createAnalyser();
      analyserNode.fftSize = 256;
      source.connect(analyserNode);
      
      dataArray = new Uint8Array(analyserNode.frequencyBinCount);
      
      // Iniciar o medidor de volume
      updateVolumeMeter();
      
      // Configurar o MediaRecorder
      mediaRecorder = new MediaRecorder(stream, { mimeType: 'audio/webm' });
      
      // Iniciar uma nova sessão no servidor
      socket.emit('start-session');
      
      // Evento para capturar os chunks de áudio
      mediaRecorder.ondataavailable = (event) => {
        if (event.data.size > 0) {
          audioChunks.push(event.data);
        }
      };
      
      // Iniciar a gravação
      mediaRecorder.start(1000); // Capturar chunks a cada 1 segundo
      isRecording = true;
      
      // Configurar o intervalo para enviar chunks ao servidor
      recordingInterval = setInterval(sendAudioChunk, 1000);
      
      // Atualizar a interface
      statusElement.textContent = 'Gravando...';
      recordBtn.disabled = true;
      stopBtn.disabled = false;
      document.body.classList.add('recording');
      
    } catch (error) {
      console.error('Erro ao iniciar gravação:', error);
      statusElement.textContent = 'Erro ao acessar microfone';
    }
  }
  
  // Parar gravação
  function stopRecording() {
    if (mediaRecorder && isRecording) {
      mediaRecorder.stop();
      isRecording = false;
      
      // Limpar o intervalo de envio de chunks
      clearInterval(recordingInterval);
      
      // Finalizar a sessão no servidor
      if (sessionId) {
        socket.emit('end-session', { sessionId });
      }
      
      // Atualizar a interface
      statusElement.textContent = 'Gravação finalizada';
      recordBtn.disabled = false;
      stopBtn.disabled = true;
      document.body.classList.remove('recording');
      
      // Parar o medidor de volume
      volumeBar.style.width = '0%';
    }
  }
  
  // Enviar chunk de áudio para o servidor
  async function sendAudioChunk() {
    if (audioChunks.length > 0 && sessionId) {
      // Criar um blob com os chunks acumulados
      const audioBlob = new Blob(audioChunks, { type: 'audio/webm' });
      audioChunks = []; // Limpar os chunks após enviar
      
      // Converter o blob para array buffer
      const arrayBuffer = await audioBlob.arrayBuffer();
      
      // Enviar o chunk para o servidor via WebSocket
      socket.emit('audio-chunk', {
        sessionId,
        audioChunk: Array.from(new Uint8Array(arrayBuffer))
      });
    }
  }
  
  // Atualizar o medidor de volume
  function updateVolumeMeter() {
    if (analyserNode && isRecording) {
      analyserNode.getByteFrequencyData(dataArray);
      
      // Calcular o volume médio
      const average = dataArray.reduce((sum, value) => sum + value, 0) / dataArray.length;
      const volume = Math.min(100, Math.max(0, average * 100 / 255));
      
      // Atualizar a barra de volume
      volumeBar.style.width = volume + '%';
      
      // Continuar atualizando
      requestAnimationFrame(updateVolumeMeter);
    }
  }
  
  // Alternar entre as abas
  function switchTab(tabId) {
    tabButtons.forEach(btn => {
      btn.classList.remove('active');
      if (btn.dataset.tab === tabId) {
        btn.classList.add('active');
      }
    });
    
    tabPanes.forEach(pane => {
      pane.classList.remove('active');
      if (pane.id === tabId + '-tab') {
        pane.classList.add('active');
      }
    });
  }
  
  // Copiar texto da aba ativa
  function copyText() {
    let textToCopy = '';
    
    // Determinar qual aba está ativa
    const activeTab = document.querySelector('.tab-btn.active').dataset.tab;
    
    switch (activeTab) {
      case 'structured':
        textToCopy = structuredText.value;
        break;
      case 'raw':
        textToCopy = rawText.value;
        break;
      case 'context':
        textToCopy = contextText.value;
        break;
    }
    
    // Copiar para a área de transferência
    navigator.clipboard.writeText(textToCopy)
      .then(() => {
        statusElement.textContent = 'Texto copiado!';
        setTimeout(() => {
          statusElement.textContent = isRecording ? 'Gravando...' : 'Pronto para gravar';
        }, 2000);
      })
      .catch(err => {
        console.error('Erro ao copiar texto:', err);
        statusElement.textContent = 'Erro ao copiar texto';
      });
  }
  
  // Limpar todos os campos de texto
  function clearText() {
    structuredText.value = '';
    rawText.value = '';
    contextText.value = '';
    statusElement.textContent = 'Texto limpo';
    setTimeout(() => {
      statusElement.textContent = isRecording ? 'Gravando...' : 'Pronto para gravar';
    }, 2000);
  }
  
  // Eventos
  recordBtn.addEventListener('click', startRecording);
  stopBtn.addEventListener('click', stopRecording);
  copyBtn.addEventListener('click', copyText);
  clearBtn.addEventListener('click', clearText);
  
  tabButtons.forEach(btn => {
    btn.addEventListener('click', () => {
      switchTab(btn.dataset.tab);
    });
  });
  
  // Inicializar a conexão WebSocket
  connectSocket();
});
