# Transcritor de Anamnese em Tempo Real

Implementação funcional de um transcritor de anamnese com transcrição em tempo real usando WebSockets e OpenAI Whisper.

## Passo a Passo de Implementação (Para Iniciantes)

### Pré-requisitos
- Node.js (versão 14 ou superior)
- NPM (gerenciador de pacotes do Node.js)
- Chave de API da OpenAI

### Estrutura de Arquivos

```
transcritor-tempo-real/
├── server/
│   ├── server.js
│   ├── routes.js
│   ├── .env
│   └── package.json
├── client/
│   ├── index.html
│   ├── style.css
│   └── script.js
└── README.md
```

## Instruções de Implementação

### 1. Configuração do Servidor

#### 1.1. Criar a estrutura de pastas
```bash
mkdir -p transcritor-tempo-real/server transcritor-tempo-real/client
cd transcritor-tempo-real/server
```

#### 1.2. Inicializar o projeto Node.js
```bash
npm init -y
```

#### 1.3. Instalar as dependências necessárias
```bash
npm install express socket.io cors dotenv openai multer uuid fs path
```

#### 1.4. Criar o arquivo .env
```bash
touch .env
```

Edite o arquivo .env e adicione sua chave de API da OpenAI:
```
OPENAI_API_KEY=sua_chave_api_aqui
PORT=3000
```

#### 1.5. Criar o arquivo server.js
```bash
touch server.js
```

Conteúdo do server.js:
```javascript
require('dotenv').config();
const express = require('express');
const http = require('http');
const socketIo = require('socket.io');
const cors = require('cors');
const path = require('path');
const fs = require('fs');
const { v4: uuidv4 } = require('uuid');
const multer = require('multer');
const { OpenAI } = require('openai');
const routes = require('./routes');

// Configuração do Express
const app = express();
app.use(cors());
app.use(express.json());
app.use(express.static(path.join(__dirname, '../client')));

// Configuração do servidor HTTP e Socket.IO
const server = http.createServer(app);
const io = socketIo(server, {
  cors: {
    origin: "*",
    methods: ["GET", "POST"]
  }
});

// Configuração do OpenAI
const openai = new OpenAI({
  apiKey: process.env.OPENAI_API_KEY,
});

// Configuração do Multer para upload de arquivos
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    const uploadDir = path.join(__dirname, 'uploads');
    if (!fs.existsSync(uploadDir)) {
      fs.mkdirSync(uploadDir);
    }
    cb(null, uploadDir);
  },
  filename: (req, file, cb) => {
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1e9);
    cb(null, uniqueSuffix + '-' + file.originalname);
  },
});
const upload = multer({ storage });

// Armazenamento de sessões de transcrição
const sessions = new Map();

// Configuração do Socket.IO
io.on('connection', (socket) => {
  console.log('Cliente conectado:', socket.id);
  
  // Criar nova sessão de transcrição
  socket.on('start-session', () => {
    const sessionId = uuidv4();
    sessions.set(sessionId, {
      transcription: '',
      organizedText: '',
      lastChunkTime: Date.now()
    });
    socket.emit('session-started', { sessionId });
    console.log('Nova sessão criada:', sessionId);
  });
  
  // Receber chunks de áudio
  socket.on('audio-chunk', async (data) => {
    try {
      const { sessionId, audioChunk } = data;
      
      if (!sessions.has(sessionId)) {
        socket.emit('error', { message: 'Sessão não encontrada' });
        return;
      }
      
      const session = sessions.get(sessionId);
      
      // Salvar o chunk de áudio temporariamente
      const chunkPath = path.join(__dirname, 'uploads', `${sessionId}-${Date.now()}.webm`);
      fs.writeFileSync(chunkPath, Buffer.from(audioChunk));
      
      // Processar o chunk com Whisper
      const transcription = await processAudioChunk(chunkPath, session.transcription);
      
      // Atualizar a transcrição da sessão
      session.transcription += ' ' + transcription.text;
      session.lastChunkTime = Date.now();
      
      // Verificar se é hora de organizar o texto (a cada 15 segundos)
      let organizedText = session.organizedText;
      if (Date.now() - session.lastOrganizeTime > 15000 || !session.lastOrganizeTime) {
        organizedText = await organizeText(session.transcription);
        session.organizedText = organizedText;
        session.lastOrganizeTime = Date.now();
      }
      
      // Enviar a transcrição atualizada para o cliente
      socket.emit('transcription-update', {
        transcription: session.transcription,
        organizedText: organizedText
      });
      
      // Limpar o arquivo temporário
      fs.unlinkSync(chunkPath);
      
    } catch (error) {
      console.error('Erro ao processar chunk de áudio:', error);
      socket.emit('error', { message: 'Erro ao processar áudio' });
    }
  });
  
  // Finalizar sessão
  socket.on('end-session', ({ sessionId }) => {
    if (sessions.has(sessionId)) {
      sessions.delete(sessionId);
      console.log('Sessão finalizada:', sessionId);
    }
  });
  
  // Desconexão
  socket.on('disconnect', () => {
    console.log('Cliente desconectado:', socket.id);
  });
});

// Função para processar chunk de áudio com Whisper
async function processAudioChunk(audioPath, previousText) {
  try {
    const transcription = await openai.audio.transcriptions.create({
      file: fs.createReadStream(audioPath),
      model: 'whisper-1',
      prompt: previousText.slice(-200), // Usar os últimos 200 caracteres como contexto
      language: 'pt'
    });
    
    return transcription;
  } catch (error) {
    console.error('Erro na transcrição:', error);
    throw error;
  }
}

// Função para organizar o texto em formato de anamnese
async function organizeText(text) {
  try {
    const prompt = `
Organize o seguinte texto de anamnese em tempo real. Mantenha a estrutura, mas não conclua seções que ainda não foram mencionadas:

"${text}"

Organize em:
1. Queixa Principal
2. História da Doença Atual
3. História Familiar
4. História Pregressa Patológica
5. Medicações de uso contínuo
6. Exame Físico
7. Diagnóstico
8. Conduta
`;
    
    const completion = await openai.chat.completions.create({
      model: 'o3-mini',
      messages: [{ role: 'user', content: prompt }],
    });
    
    return completion.choices[0].message.content;
  } catch (error) {
    console.error('Erro ao organizar texto:', error);
    return text; // Retorna o texto original em caso de erro
  }
}

// Rotas da API
app.use('/', routes);

// Iniciar o servidor
const PORT = process.env.PORT || 3000;
server.listen(PORT, () => {
  console.log(`Servidor rodando na porta ${PORT}`);
});
```

#### 1.6. Criar o arquivo routes.js
```bash
touch routes.js
```

Conteúdo do routes.js:
```javascript
const express = require('express');
const path = require('path');
const router = express.Router();

// Rota principal - serve o cliente
router.get('/', (req, res) => {
  res.sendFile(path.join(__dirname, '../client/index.html'));
});

// Rota de verificação de saúde
router.get('/health', (req, res) => {
  return res.json({ status: 'ok' });
});

module.exports = router;
```

### 2. Configuração do Cliente

#### 2.1. Criar os arquivos do cliente
```bash
cd ../client
touch index.html style.css script.js
```

#### 2.2. Conteúdo do index.html
```html
<!DOCTYPE html>
<html lang="pt-BR">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Transcritor de Anamnese em Tempo Real</title>
  <link rel="stylesheet" href="style.css">
</head>
<body>
  <div class="container">
    <h1>Transcritor de Anamnese em Tempo Real</h1>
    
    <div class="status-bar">
      <div id="status">Pronto para gravar</div>
      <div class="volume-meter">
        <div id="volume-bar"></div>
      </div>
    </div>
    
    <div class="controls">
      <button id="record-btn">Gravar</button>
      <button id="stop-btn" disabled>Parar</button>
    </div>
    
    <div class="tabs">
      <div class="tab-buttons">
        <button class="tab-btn active" data-tab="structured">Anamnese Estruturada</button>
        <button class="tab-btn" data-tab="raw">Transcrição Completa</button>
        <button class="tab-btn" data-tab="context">Contexto Clínico</button>
      </div>
      
      <div class="tab-content">
        <div id="structured-tab" class="tab-pane active">
          <textarea id="structured-text" placeholder="Anamnese estruturada aparecerá aqui..." readonly></textarea>
        </div>
        <div id="raw-tab" class="tab-pane">
          <textarea id="raw-text" placeholder="Transcrição completa aparecerá aqui..." readonly></textarea>
        </div>
        <div id="context-tab" class="tab-pane">
          <textarea id="context-text" placeholder="Adicione informações de contexto clínico aqui..."></textarea>
        </div>
      </div>
    </div>
    
    <div class="actions">
      <button id="copy-btn">Copiar</button>
      <button id="clear-btn">Limpar</button>
    </div>
  </div>
  
  <script src="https://cdn.socket.io/4.5.4/socket.io.min.js"></script>
  <script src="script.js"></script>
</body>
</html>
```

#### 2.3. Conteúdo do style.css
```css
* {
  box-sizing: border-box;
  margin: 0;
  padding: 0;
}

body {
  font-family: Arial, sans-serif;
  background-color: #26282b;
  color: #ffffff;
  line-height: 1.6;
}

.container {
  max-width: 800px;
  margin: 0 auto;
  padding: 20px;
}

h1 {
  text-align: center;
  margin-bottom: 20px;
}

.status-bar {
  display: flex;
  align-items: center;
  justify-content: space-between;
  background-color: rgba(49, 52, 56, 0.6);
  padding: 10px;
  border-radius: 5px;
  margin-bottom: 20px;
}

#status {
  font-weight: bold;
}

.volume-meter {
  width: 200px;
  height: 10px;
  background-color: rgba(68, 72, 77, 0.8);
  border-radius: 5px;
  overflow: hidden;
}

#volume-bar {
  height: 100%;
  width: 0;
  background-color: #47ffc4;
  transition: width 0.1s;
}

.controls {
  display: flex;
  justify-content: center;
  gap: 10px;
  margin-bottom: 20px;
}

button {
  padding: 10px 20px;
  border: none;
  border-radius: 5px;
  background-color: #47ffc4;
  color: #26282b;
  font-weight: bold;
  cursor: pointer;
}

button:disabled {
  background-color: #cccccc;
  cursor: not-allowed;
}

.tabs {
  margin-bottom: 20px;
}

.tab-buttons {
  display: flex;
  margin-bottom: 10px;
}

.tab-btn {
  flex: 1;
  background-color: rgba(49, 52, 56, 0.6);
  color: #ffffff;
}

.tab-btn.active {
  background-color: #47ffc4;
  color: #26282b;
}

.tab-pane {
  display: none;
}

.tab-pane.active {
  display: block;
}

textarea {
  width: 100%;
  height: 300px;
  padding: 10px;
  border-radius: 5px;
  background-color: rgba(49, 52, 56, 0.6);
  color: #ffffff;
  border: 1px solid rgba(71, 255, 196, 0.1);
  resize: vertical;
}

.actions {
  display: flex;
  justify-content: center;
  gap: 10px;
}

.recording #record-btn {
  background-color: #ff4500;
}
```

#### 2.4. Conteúdo do script.js
```javascript
document.addEventListener('DOMContentLoaded', () => {
  // Elementos da interface
  const recordBtn = document.getElementById('record-btn');
  const stopBtn = document.getElementById('stop-btn');
  const statusElement = document.getElementById('status');
  const volumeBar = document.getElementById('volume-bar');
  const structuredText = document.getElementById('structured-text');
  const rawText = document.getElementById('raw-text');
  const contextText = document.getElementById('context-text');
  const copyBtn = document.getElementById('copy-btn');
  const clearBtn = document.getElementById('clear-btn');
  const tabButtons = document.querySelectorAll('.tab-btn');
  const tabPanes = document.querySelectorAll('.tab-pane');
  
  // Variáveis para gravação e WebSocket
  let mediaRecorder;
  let audioChunks = [];
  let socket;
  let sessionId;
  let recordingInterval;
  let analyserNode;
  let dataArray;
  let isRecording = false;
  
  // Conectar ao servidor WebSocket
  function connectSocket() {
    socket = io('http://localhost:3000');
    
    socket.on('connect', () => {
      console.log('Conectado ao servidor WebSocket');
      statusElement.textContent = 'Conectado ao servidor';
    });
    
    socket.on('session-started', (data) => {
      sessionId = data.sessionId;
      console.log('Sessão iniciada:', sessionId);
    });
    
    socket.on('transcription-update', (data) => {
      rawText.value = data.transcription;
      if (data.organizedText) {
        structuredText.value = data.organizedText;
      }
      
      // Auto-scroll para a parte inferior
      rawText.scrollTop = rawText.scrollHeight;
      structuredText.scrollTop = structuredText.scrollHeight;
    });
    
    socket.on('error', (error) => {
      console.error('Erro WebSocket:', error);
      statusElement.textContent = 'Erro: ' + error.message;
    });
    
    socket.on('disconnect', () => {
      console.log('Desconectado do servidor WebSocket');
      statusElement.textContent = 'Desconectado do servidor';
    });
  }
  
  // Iniciar gravação
  async function startRecording() {
    try {
      // Solicitar permissão para acessar o microfone
      const stream = await navigator.mediaDevices.getUserMedia({ audio: true });
      
      // Configurar o analisador de áudio para o medidor de volume
      const audioContext = new AudioContext();
      const source = audioContext.createMediaStreamSource(stream);
      analyserNode = audioContext.createAnalyser();
      analyserNode.fftSize = 256;
      source.connect(analyserNode);
      
      dataArray = new Uint8Array(analyserNode.frequencyBinCount);
      
      // Iniciar o medidor de volume
      updateVolumeMeter();
      
      // Configurar o MediaRecorder
      mediaRecorder = new MediaRecorder(stream, { mimeType: 'audio/webm' });
      
      // Iniciar uma nova sessão no servidor
      socket.emit('start-session');
      
      // Evento para capturar os chunks de áudio
      mediaRecorder.ondataavailable = (event) => {
        if (event.data.size > 0) {
          audioChunks.push(event.data);
        }
      };
      
      // Iniciar a gravação
      mediaRecorder.start(1000); // Capturar chunks a cada 1 segundo
      isRecording = true;
      
      // Configurar o intervalo para enviar chunks ao servidor
      recordingInterval = setInterval(sendAudioChunk, 1000);
      
      // Atualizar a interface
      statusElement.textContent = 'Gravando...';
      recordBtn.disabled = true;
      stopBtn.disabled = false;
      document.body.classList.add('recording');
      
    } catch (error) {
      console.error('Erro ao iniciar gravação:', error);
      statusElement.textContent = 'Erro ao acessar microfone';
    }
  }
  
  // Parar gravação
  function stopRecording() {
    if (mediaRecorder && isRecording) {
      mediaRecorder.stop();
      isRecording = false;
      
      // Limpar o intervalo de envio de chunks
      clearInterval(recordingInterval);
      
      // Finalizar a sessão no servidor
      if (sessionId) {
        socket.emit('end-session', { sessionId });
      }
      
      // Atualizar a interface
      statusElement.textContent = 'Gravação finalizada';
      recordBtn.disabled = false;
      stopBtn.disabled = true;
      document.body.classList.remove('recording');
      
      // Parar o medidor de volume
      volumeBar.style.width = '0%';
    }
  }
  
  // Enviar chunk de áudio para o servidor
  async function sendAudioChunk() {
    if (audioChunks.length > 0 && sessionId) {
      // Criar um blob com os chunks acumulados
      const audioBlob = new Blob(audioChunks, { type: 'audio/webm' });
      audioChunks = []; // Limpar os chunks após enviar
      
      // Converter o blob para array buffer
      const arrayBuffer = await audioBlob.arrayBuffer();
      
      // Enviar o chunk para o servidor via WebSocket
      socket.emit('audio-chunk', {
        sessionId,
        audioChunk: Array.from(new Uint8Array(arrayBuffer))
      });
    }
  }
  
  // Atualizar o medidor de volume
  function updateVolumeMeter() {
    if (analyserNode && isRecording) {
      analyserNode.getByteFrequencyData(dataArray);
      
      // Calcular o volume médio
      const average = dataArray.reduce((sum, value) => sum + value, 0) / dataArray.length;
      const volume = Math.min(100, Math.max(0, average * 100 / 255));
      
      // Atualizar a barra de volume
      volumeBar.style.width = volume + '%';
      
      // Continuar atualizando
      requestAnimationFrame(updateVolumeMeter);
    }
  }
  
  // Alternar entre as abas
  function switchTab(tabId) {
    tabButtons.forEach(btn => {
      btn.classList.remove('active');
      if (btn.dataset.tab === tabId) {
        btn.classList.add('active');
      }
    });
    
    tabPanes.forEach(pane => {
      pane.classList.remove('active');
      if (pane.id === tabId + '-tab') {
        pane.classList.add('active');
      }
    });
  }
  
  // Copiar texto da aba ativa
  function copyText() {
    let textToCopy = '';
    
    // Determinar qual aba está ativa
    const activeTab = document.querySelector('.tab-btn.active').dataset.tab;
    
    switch (activeTab) {
      case 'structured':
        textToCopy = structuredText.value;
        break;
      case 'raw':
        textToCopy = rawText.value;
        break;
      case 'context':
        textToCopy = contextText.value;
        break;
    }
    
    // Copiar para a área de transferência
    navigator.clipboard.writeText(textToCopy)
      .then(() => {
        statusElement.textContent = 'Texto copiado!';
        setTimeout(() => {
          statusElement.textContent = isRecording ? 'Gravando...' : 'Pronto para gravar';
        }, 2000);
      })
      .catch(err => {
        console.error('Erro ao copiar texto:', err);
        statusElement.textContent = 'Erro ao copiar texto';
      });
  }
  
  // Limpar todos os campos de texto
  function clearText() {
    structuredText.value = '';
    rawText.value = '';
    contextText.value = '';
    statusElement.textContent = 'Texto limpo';
    setTimeout(() => {
      statusElement.textContent = isRecording ? 'Gravando...' : 'Pronto para gravar';
    }, 2000);
  }
  
  // Eventos
  recordBtn.addEventListener('click', startRecording);
  stopBtn.addEventListener('click', stopRecording);
  copyBtn.addEventListener('click', copyText);
  clearBtn.addEventListener('click', clearText);
  
  tabButtons.forEach(btn => {
    btn.addEventListener('click', () => {
      switchTab(btn.dataset.tab);
    });
  });
  
  // Inicializar a conexão WebSocket
  connectSocket();
});
```

### 3. Executando a Aplicação

#### 3.1. Iniciar o servidor
```bash
cd ../server
node server.js
```

#### 3.2. Acessar a aplicação
Abra seu navegador e acesse:
```
http://localhost:3000
```

## Instruções de Uso

1. Abra a aplicação no navegador
2. Clique no botão "Gravar" para iniciar a gravação
3. Fale normalmente - a transcrição aparecerá em tempo real
4. A anamnese estruturada será atualizada periodicamente
5. Clique em "Parar" para finalizar a gravação
6. Use as abas para alternar entre a anamnese estruturada e a transcrição completa
7. Use o botão "Copiar" para copiar o texto da aba atual
8. Use o botão "Limpar" para limpar todos os campos de texto

## Solução de Problemas

### Erro de conexão com o servidor
- Verifique se o servidor está rodando na porta 3000
- Verifique se não há outro serviço usando a mesma porta

### Erro ao acessar o microfone
- Certifique-se de que o navegador tem permissão para acessar o microfone
- Tente usar o Chrome, que tem melhor suporte para APIs de áudio

### Erro na transcrição
- Verifique se a chave da API da OpenAI está correta no arquivo .env
- Verifique se há créditos suficientes na sua conta da OpenAI

### Latência alta
- Reduza o tamanho dos chunks de áudio (altere o valor no MediaRecorder.start())
- Verifique sua conexão com a internet

## Próximos Passos

1. Adicionar autenticação de usuários
2. Implementar armazenamento persistente das transcrições
3. Melhorar a interface do usuário
4. Adicionar suporte para múltiplos idiomas
5. Implementar uma versão offline usando Whisper.cpp
